
namespace org.owasp.appsensor.reference
{

    using System;
    using System.Collections.Generic;

    /**
     * NOTE - The locking employed in the class isn't really suitable for a proper multithreaded distributed
     * environment.
     * @author luke b
     *
     */
    public class DefaultServiceStore : ServiceStore
    {

        /** Hashtable of path to AppSensorServiceMonitor object, represented by the request URI in the default implementation */
        private readonly Dictionary<String, AppSensorServiceMonitor> serviceStorage = new Dictionary<String, AppSensorServiceMonitor>();
        /** Hashtable of path to AppSensorServiceMonitor object, represented by the request URI / user id in the default implementation */
        private readonly Dictionary<String, AppSensorServiceMonitor> perUserServiceStorage = new Dictionary<String, AppSensorServiceMonitor>();

        /**
         * Retrieves the AppSensorServiceMonitor object for the service specified, creates an object if doesn't exist
         * @param service name of the service to retrieve 
         * @return AppSensorServiceMonitor object for the service specified
         */
        public AppSensorServiceMonitor GetService(String service)
        {
            AppSensorServiceMonitor serviceMon = serviceStorage[service];
            if (serviceMon == null)
            {
                serviceMon = new AppSensorServiceMonitor(service, null);
                serviceStorage.Add(service, serviceMon);
            }
            return serviceMon;
        }

        /**
         * Retrieves the AppSensorServiceMonitor object for the service/userId specified, creates an object if doesn't exist
         * @param service name of the service to retrieve 
         * @param userId userId of the user to retrieve
         * @return AppSensorServiceMonitor object for the service/userId specified
         */
        public AppSensorServiceMonitor GetServiceForUser(String service, String userId)
        {
            String key = GetServiceKey(service, userId);
            AppSensorServiceMonitor serviceMon = perUserServiceStorage[key];
            if (serviceMon == null)
            {
                serviceMon = new AppSensorServiceMonitor(service, userId);
                perUserServiceStorage.Add(key, serviceMon);
            }
            return serviceMon;
        }

        public void Save(AppSensorServiceMonitor assm)
        {
            // Nothing to do here as this implementation is entirely in process.
            // If this were a distributed system like membase or mysql we'd write to that.
        }

        /**
         * Helper method to create concatenated service/user string to represent the 
         * unique key for service/user storage
         * @param service name of the service 
         * @param userId name of the userId
         * @return concatenated service/user string to represent the unique key for service/user storage
         */
        private static String GetServiceKey(String service, String userId)
        {
            if (userId != null)
            {
                return service + "--" + userId;
            }
            return service;
        }
    }
}